/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xusb.c 
 * -----------------------------------------------------------------*/

#include <stdio.h>
#include <windows.h>
#include <winioctl.h>

#include <xwdmusb.h>

#include <xusb.h> 
#include <xlasterr.h>
#include <xidentos.h> 
#include <timeout.h>

#include <setupapi.h>
#include <regstr.h>

#include <initguid.h>
#include <xguids.h>

#include <assert.h>

#include <xezusbsy.h>

/* 23er only ; keep konsistent with FW (vendcmd.c) */
#define ANCHOR_USB_DRR_0              0xB4
#define ANCHOR_USB_DRR_1              0xB5
#define ANCHOR_USB_DRW_0              0xB6
#define ANCHOR_USB_DRW_1              0xB7

#define BESTX_FX2HEX_LOAD
#ifdef BESTX_FX2HEX_LOAD
#define MAX_EP0_XFER_SIZE 0x1000   /* 4K Control EP0 transfer limit imposed by OS */
#define TGT_IMG_SIZE      0x10000  /* 64KB (65,536 Byte) target image */
#define TGT_SEG_SIZE      0x10     /* 16 byte segments */

/* A memory segment from .hex file */
typedef struct _MemSeg
{
  DWORD TAddr; /* Target Address */
  DWORD HAddr; /* Host Address   */
  DWORD Size;  /* block size     */
  DWORD Stat;  /* status         */
  PCHAR pData; /* data bytes     */
} MemSeg;


/* Holds the binary image for FX2 FW */
typedef struct _TMemImg
{
  char data[TGT_IMG_SIZE];   /* target image store */
} TMemImg;


typedef struct _TMemCache
{
  TMemImg* pImg; /* pointer to image */
  int nSeg;      /* segment count */
  MemSeg pSeg[TGT_IMG_SIZE/TGT_SEG_SIZE]; /* info about segments */
} TMemCache;

/* --------------------------------------------------------------------------
 * STATICS
 * -------------------------------------------------------------------------- */
static int intel_in(FILE *fpIn, TMemCache* pMemCache, DWORD ioOffset, char endianFlags, BOOLEAN spaces);
static bx_errtype AnchorDownload(HANDLE OsHandle, char *pData, bx_int32 numBytes, bx_int16 memOffset);
static bx_errtype Reset(HANDLE OsHandle,bx_int8 hold);
#endif

#define MAX_NAME_LENGTH 1024

/*---------------------------------------------------------------------------*
 * BestXUSBDeviceNameGet
 *
 * Purpose: Gets the name that we'll use to open a USB device 
 *---------------------------------------------------------------------------*/

#if defined(WIN2K) || defined(WIN95)
static bx_errtype BestXUSBDeviceNameGet
(
  bx_int32 CardIndex,   /* <IN> zero based card index                */
  char *   DeviceName,  /* <OUT> Name of device for CreateFile()     */
  char *   FriendlyName /* <OUT> Friendly name resp. desc. of device */
) 
{
  HDEVINFO ClassInfo;              /* General info about BEST interface        */
  SP_DEVICE_INTERFACE_DATA IFData; /* Interface data: Info about one BEST card */
  PSP_INTERFACE_DEVICE_DETAIL_DATA DetailIFData; /* Some more details          */
  SP_DEVINFO_DATA DeviceInfoData = {sizeof(SP_DEVINFO_DATA)}; /* Device information */
  DWORD NumBytes;

  GUID MyGUID=GUID_BESTX;
  
  if (CardIndex<BX_USB_PORTNUM_OFFSET)
  {
    /* Open FX2 */
    BESTX_SPRINTF(DeviceName,"\\\\.\\b_usbfx-%lu",CardIndex);  
    return BX_E_OK;
  }
  else
  {
    /* Open D12 */
    CardIndex-=BX_USB_PORTNUM_OFFSET;
  }

  IFData.cbSize = sizeof(IFData);


  /* Open an enumeration handle so we can locate the devices of the BEST class */

  ClassInfo = SetupDiGetClassDevs(&MyGUID, NULL, NULL, DIGCF_PRESENT | DIGCF_INTERFACEDEVICE);
  if (ClassInfo == INVALID_HANDLE_VALUE)
  {
    /* chris TBD good error message */
    return BX_E_ERROR;
  }
  
  /* Try to get info about the wanted card */
  if (!SetupDiEnumDeviceInterfaces(ClassInfo,NULL,&MyGUID,CardIndex,&IFData))
  {
    /* The device does not exist */
    /* chris TBD good error message */
    SetupDiDestroyDeviceInfoList(ClassInfo);
    return BX_E_ERROR;
  }

  /* Determine the symbolic link name for this device instance. 
     Since this is variable in length, make an initial call to determine
     the required length.
  */

  SetupDiGetDeviceInterfaceDetail(ClassInfo, &IFData, NULL, 0, &NumBytes, NULL);
  DetailIFData = (PSP_INTERFACE_DEVICE_DETAIL_DATA) BestXMemMalloc(NumBytes);
  if (DetailIFData==NULL)
  {
    SetupDiDestroyDeviceInfoList(ClassInfo);
    return BX_E_HOST_MEM_FULL;
  }


  DetailIFData->cbSize = sizeof(SP_INTERFACE_DEVICE_DETAIL_DATA);

  if (!SetupDiGetDeviceInterfaceDetail(ClassInfo,&IFData,DetailIFData,NumBytes,NULL, &DeviceInfoData))
  {
    /* Can't get detail info */
    BestXMemFree(&((PVOID)DetailIFData));
    SetupDiDestroyDeviceInfoList(ClassInfo);
    /* chris TBD good error message */
    return BX_E_ERROR;
  }

  if (DeviceName)
  {
    BESTX_STRNCPY(DeviceName,DetailIFData->DevicePath,MAX_NAME_LENGTH);
  }

  /* Determine the device's friendly name */
  if (FriendlyName 
      
      &&
      
      !SetupDiGetDeviceRegistryProperty(
       ClassInfo,
       &DeviceInfoData,
       SPDRP_FRIENDLYNAME,
       NULL,
       (PBYTE) FriendlyName,
       MAX_NAME_LENGTH,
       NULL)
       
       && 
       
       !SetupDiGetDeviceRegistryProperty(
       ClassInfo,
       &DeviceInfoData,
       SPDRP_DEVICEDESC,
       NULL,
       (PBYTE) FriendlyName,
       MAX_NAME_LENGTH,
       NULL))
  {
    /* No friendly name, no description: Let's take the device name */
    BESTX_STRNCPY(FriendlyName, DetailIFData->DevicePath, MAX_NAME_LENGTH);
  }

  SetupDiDestroyDeviceInfoList(ClassInfo);

  return BX_E_OK;
}

#endif

/* --------------------------------------------------------------------------
 * CAPI-only functions 
 * -------------------------------------------------------------------------- */

/*---------------------------------------------------------------------------*
 * BestXOpenUSB
 *
 * Purpose: Open a USB card device 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXOpenUSB(
  bx_int32 dwCardId, 
  bx_portnumtype * pOsHandle,
  bx_charptrtype filename)

{
  HANDLE OsHandle;

#ifdef BESTX_FX2HEX_LOAD
  FILE *fp;
  TMemCache m_MemCache;    /* target mem cache info */
  TMemImg m_MemImg;        /* target memory image */
  int read,i;
  int maxtries;
#endif

  char cDeviceName[MAX_NAME_LENGTH];
  char cDeviceFriendlyName[MAX_NAME_LENGTH];
  bx_errtype err;

  assert(pOsHandle && "Null ptr. passed to BestXOpenUSB()"); 

  /* init for failure */
  *pOsHandle = INVALID_OS_HANDLE;

  
  /* get the Win32 name that we'll use to open the device. */
  if (BX_E_OK != (err = BestXUSBDeviceNameGet(dwCardId, cDeviceName,cDeviceFriendlyName)))
  {
    return err;
  }

  /* now try to open the device */
  OsHandle = CreateFile(cDeviceName,
                        GENERIC_WRITE | GENERIC_READ,
                        0, 
                        NULL,
                        OPEN_EXISTING,
                        0,
                        NULL);


  if (INVALID_HANDLE_VALUE == OsHandle)
  {
    DBG_ApiLastError;
    return BX_E_USB_OPEN;
  }                       

#ifdef BESTX_FX2HEX_LOAD
  if (dwCardId<BX_USB_PORTNUM_OFFSET && filename)
  {
    /* Load FX2 FW from file to card */

    /* 1. Read FW from .hex file and convert into binary segments
      2. Put CPU into reset 
      3. Download binary segments of FW
      4. Run FW by taking CPU out of reset
     */

    /* Open FW file */
    if ((fp = fopen(filename,"r")) == NULL)
    {
      return BX_E_FILE_ITEM_NOT_FOUND;
    }

    /* Convert FW-ASCII data from .hex file into binary segments */
    m_MemCache.pImg = &m_MemImg;
    read = intel_in(
                        fp,
                        &m_MemCache,
                        0, /* offset; set to -1 to get offset from srec or hex file */
                        0, /* Mask for endianConversion */
                        FALSE
                      );
    fclose(fp);

    if (read<1)
    {
      return BX_E_ERROR;
    }

    /* Put CPU into reset */
    err=Reset(OsHandle,1);
    if (err!=BX_E_OK) 
    {
      return err;
    }

    /* Download FW */
    for(i=0; i<m_MemCache.nSeg; i++)
    {
      err = AnchorDownload(OsHandle,m_MemCache.pSeg[i].pData,m_MemCache.pSeg[i].Size,(USHORT)m_MemCache.pSeg[i].TAddr);
      if (err!=BX_E_OK) 
      {
        return err;
      }
    }

    /* Get CPU out of reset -> CPU runs downloaded FW */
    err=Reset(OsHandle,0);

    if (err!=BX_E_OK) 
    {
      return err;
    }

    /* Close connection.
      Caution: If the time between the previous Reset() 
      and the following BestXCloseUSB() gets too long (~second)
      (e.g. when setting breakpoint in debugger), the
      FX2 renumerates and comes up with ID=dwCardId+1
      instead of ID=dwCardId, because the old device
      is not yet closed. This would lead to a failure
      of CreateFile() */

    BestXCloseUSB(OsHandle);
    
    /* Wait for renumeration to finish.
      Typically we need to wait between
      2 and 3 seconds before we can call
      CreateFile() successfully again.
      Caution: If you remove the following Sleep(),
      CreateFile() will succeed, because renumeration
      has not yet begun, but the connection then
      becomes invalid and leads to errors when accessing
      the device later. */
   
    BESTX_SLEEP(4000);

    /* Typically renumeration has begun here, i.e. we are
      disconnected but not yet reconnected again.
    */

    maxtries=30; /* typically we need 6 retries here */    
    OsHandle=INVALID_HANDLE_VALUE;
    do
    {
      OsHandle = CreateFile(cDeviceName,
                          GENERIC_WRITE|GENERIC_READ,
                          0,
                          NULL,
                          OPEN_EXISTING,
                          0,
                          NULL);

      BESTX_SLEEP(100);
      maxtries--;
    } while (OsHandle==INVALID_HANDLE_VALUE && maxtries);


    if (INVALID_HANDLE_VALUE == OsHandle)
    {
      DBG_ApiLastError;
      return BX_E_USB_OPEN;
    }                       
  } /* if FX2 */
#endif

  *pOsHandle = (bx_portnumtype) OsHandle;

  return err;
}

#ifdef BESTX_FX2HEX_LOAD

#define MAXSTR 256 /* Maximum length of Intel Hex file string */

/* This functions converts S-Records from .hex file into binary segments */
static int intel_in(FILE *fpIn, TMemCache* pMemCache, DWORD ioOffset, 
                    char endianFlags, BOOLEAN spaces)
{
  int i;
  char str[MAXSTR];
  unsigned byte;

  int curSeg = 0;  /* current seg record */
  int recType;
  unsigned addr;
  int cnt;
  unsigned int totalRead = 0;
  int CNTFIELD ;
  int ADDRFIELD;
  int RECFIELD ;
  int DATAFIELD;
  PCHAR ptr;

  /* offsets of fields within record -- may change later due to "spaces" setting */
  CNTFIELD    = 1;
  ADDRFIELD   = 3;
  RECFIELD    = 7;
  DATAFIELD   = 9;

  if (!fpIn)
    return(0);

  addr = 0;

  pMemCache->nSeg = 0;
  while(fgets(str,MAXSTR,fpIn))
  {
    if(str[0]!=':')
    {
      return(-1);
    }

    /* get the record type */
    if (spaces || str[1] == ' ')
    {
      CNTFIELD    = 1 + 1;
      ADDRFIELD   = 3 + 2;
      RECFIELD    = 7 + 3;
      DATAFIELD   = 9 + 4;
    }

    sscanf(str+RECFIELD,"%2x",&recType);

    ptr = (PCHAR)pMemCache->pImg;
    switch(recType)
    {
    case 2: /*seg record*/
      sscanf(str+DATAFIELD,"%4x",&curSeg);
      curSeg *= 0x10;
      break;

    case 0: /*data record*/
      sscanf(str+CNTFIELD,"%2x",&cnt);
      sscanf(str+ADDRFIELD,"%4x",&addr);
      if(addr >= TGT_IMG_SIZE)
      {
        /* theApp.report_error("Error loading file: address out of range\n"); */
        return(totalRead);
      }
      ptr += addr; /* get pointer to location in image */

      if(pMemCache->nSeg && 
        (pMemCache->pSeg[pMemCache->nSeg-1].TAddr == 
        addr - pMemCache->pSeg[pMemCache->nSeg-1].Size) &&
        (pMemCache->pSeg[pMemCache->nSeg-1].Size + cnt <= MAX_EP0_XFER_SIZE) )
      { 
        /* if the segment is contiguous to the last segment, and it's not too big yet */
        pMemCache->pSeg[pMemCache->nSeg-1].Size += cnt; /* append to previous segment */
      }
      else
      { 
        /* start a new segment */
        pMemCache->pSeg[pMemCache->nSeg].TAddr = addr;
        pMemCache->pSeg[pMemCache->nSeg].Size = cnt;
        pMemCache->pSeg[pMemCache->nSeg].pData = ptr;
        pMemCache->nSeg++;
      }


      for(i=0; i<cnt; i++)
      {
        sscanf(str+DATAFIELD+i*2,"%2x",&byte);
        *(ptr + i) = byte;
        totalRead++;
      }
      break;

    case 1: /*end record*/
      return(totalRead);
      break;

    default:
      break;
    }
  }
  return(-1);  /* missing end record */
}

/* Downloads some bytes to the specified location in FX2 chip */
static bx_errtype AnchorDownload(HANDLE OsHandle, char *pData, bx_int32 numBytes, bx_int16 memOffset)
{
  BOOLEAN bresult = FALSE;
  
  bx_int32 bytesWritten=0;

  ANCHOR_DOWNLOAD_CONTROL myDownload;

  myDownload.Offset = memOffset;


  bresult = (BOOLEAN) DeviceIoControl (OsHandle,
            IOCTL_EZUSB_ANCHOR_DOWNLOAD, 
            &myDownload,
            sizeof(ANCHOR_DOWNLOAD_CONTROL),
            (void *) pData,
            numBytes,
            &bytesWritten,
            NULL);

  
  /* Note: bytesWritten is always 0!! even with successful D/L ???? */
  if (bresult != TRUE) 
  {
    return BX_E_ERROR;
  } 

  return BX_E_OK;
}


/* Puts FX2's 8051 CPU into reset resp. releases reset again */
static bx_errtype Reset(HANDLE OsHandle, bx_int8 hold)
{
  VENDOR_REQUEST_IN myRequest;
  BOOLEAN bresult = FALSE;
  bx_int32 nBytes = 0;

  myRequest.bRequest  = ANCHOR_LOAD_INTERNAL;   /* A0: Firmware Load (handled by core) */
  myRequest.wValue    = CPUCS_REG_FX2;          /* CPUCS at 0xE600 */
  myRequest.wIndex    = 0x00;
  myRequest.wLength   = 0x01;
  myRequest.bData     = hold; /* 1=CPU Reset; 0=bring CPU out of reset, i.e. run */
  myRequest.direction = 0x00;
          
  bresult = (BOOLEAN) DeviceIoControl (OsHandle,
              IOCTL_Ezusb_VENDOR_REQUEST,
              &myRequest,
              sizeof(VENDOR_REQUEST_IN),
              NULL,
              0,
              &nBytes,
              NULL);

  if (bresult != TRUE)
  {
    return BX_E_ERROR;
  }
  return BX_E_OK;
}
#endif /* #ifdef BESTX_FX2HEX_LOAD */


bx_errtype EXPORT BestXUSBFXDeviceConnect(bx_portnumtype OsHandle)
{
  VENDOR_REQUEST_IN myRequest;
  BOOLEAN bresultOK = FALSE;
  bx_int32 nBytes = 0;

  myRequest.bRequest  = ANCHOR_USB_CONNECT;
  myRequest.wValue    = 0;   /* dummy */
  myRequest.wIndex    = 0x00;
  myRequest.wLength   = 0x01;
  myRequest.bData     = 0;
  myRequest.direction = 0x01; /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
              IOCTL_Ezusb_VENDOR_REQUEST,
              &myRequest,
              sizeof(VENDOR_REQUEST_IN),
              NULL,
              0,
              &nBytes,
              NULL);

  DBG_ApiFailureIsZero(bresultOK);

  if (!bresultOK)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}

void EXPORT BestXUSBFXReleaseConnection(bx_portnumtype OsHandle)
{
  VENDOR_REQUEST_IN myRequest;
  BOOLEAN bresultOK = FALSE;
  bx_int32 nBytes = 0;

  myRequest.bRequest  = ANCHOR_USB_DISCONNECT; 
  myRequest.wValue    = 0;   /* dummy */
  myRequest.wIndex    = 0x00;
  myRequest.wLength   = 0x01;
  myRequest.bData     = 0;
  myRequest.direction = 0x01; /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
              IOCTL_Ezusb_VENDOR_REQUEST,
              &myRequest,
              sizeof(VENDOR_REQUEST_IN),
              NULL,
              0,
              &nBytes,
              NULL);

  DBG_ApiFailureIsZero(bresultOK);
}

bx_errtype EXPORT BestXUSBFXCheckConnection(bx_portnumtype OsHandle)
{
  /* Returns BX_E_OK in case of valid connection */

  VENDOR_REQUEST_IN myRequest;
  BOOLEAN resultOK = FALSE;
  bx_int32 nBytes = 0;
  char fIsConnected = 0;

  myRequest.bRequest  = ANCHOR_USB_ISCONNECTED; 
  myRequest.wValue    = 0;   /* dummy */
  myRequest.wIndex    = 0x00;
  myRequest.wLength   = 0x01;
  myRequest.bData     = 0;
  myRequest.direction = 0x01; /* IN */
          


  /* DeviceIoControl() returns zero in case of an error */
  resultOK = (BOOLEAN) DeviceIoControl (OsHandle,
              IOCTL_Ezusb_VENDOR_REQUEST,
              &myRequest,
              sizeof(VENDOR_REQUEST_IN),
              &fIsConnected,              /* Buffer from driver.  */
              sizeof(char),               /* Length of buffer in bytes. */
              &nBytes,
              NULL);


  DBG_ApiFailureIsZero(resultOK);

  if (!resultOK || !fIsConnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}

bx_errtype BestXUSBFXIsDisconnected (bx_portnumtype OsHandle)
{
  /* Returns BX_E_NOT_CONNECTED, if everything is  fine and we are not connected.
     else BX_E_OK

     NOTE:
     This IO control is typically called from WaitForClose which will wait
     until this function returns BX_E_NOT_CONNECTED. In the error case this
     will result in the BX_E_OK which gives the card enough time to finish
     this cycle 
  */


  VENDOR_REQUEST_IN myRequest;
  BOOLEAN resultOK = FALSE;
  bx_int32 nBytes = 0;
  char fIsDisconnected = 0;

  myRequest.bRequest  = ANCHOR_USB_ISDISCONNECTED; 
  myRequest.wValue    = 0;   /* dummy */
  myRequest.wIndex    = 0x00;
  myRequest.wLength   = 0x01;
  myRequest.bData     = 0;
  myRequest.direction = 0x01; /* IN */
          
  /* DeviceIoControl() returns zero in case of an error */
  resultOK = (BOOLEAN) DeviceIoControl (OsHandle,
              IOCTL_Ezusb_VENDOR_REQUEST,
              &myRequest,
              sizeof(VENDOR_REQUEST_IN),
              &fIsDisconnected,           /* Buffer from driver.  */
              sizeof(char),               /* Length of buffer in bytes. */
              &nBytes,
              NULL);


  DBG_ApiFailureIsZero(resultOK);

  if (resultOK && fIsDisconnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}

/* #else */

/* Code for Philips USB chip USBD12 */

bx_errtype EXPORT BestXUSBDeviceConnect(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_USB_CONNECT,  /* IO Control code for USB card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    NULL,                       /* Buffer from driver.  */
                    0,                          /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);
  
  return BX_E_OK;
}


/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

void EXPORT BestXUSBReleaseConnection(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_USB_DISCONNECT,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    NULL,                       /* Buffer from driver.  */
                    0,                          /* Length of buffer in bytes. */
                    &ReturnedLength,            /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  return;
}

/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */


bx_errtype EXPORT BestXUSBCheckConnection(bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;
  char fIsConnected = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_USB_ISCONNECTED,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    &fIsConnected,              /* Buffer from driver.  */
                    sizeof(char),                /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess || !fIsConnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

bx_errtype BestXUSBIsDisconnected (bx_portnumtype OsHandle)
{
  BOOL fIoCtlSuccess;
  DWORD   dwReturned=0;
  int fIsDisconnected = 0;

  fIoCtlSuccess = DeviceIoControl(
                    (HANDLE) OsHandle,          /* Handle to device  */
                    (ULONG) IOCTL_USB_ISDISCONNECTED,  /* IO Control code for F-HIF card */
                    NULL,                       /* Buffer to driver. */
                    0,                          /* Length of buffer in bytes. */
                    &fIsDisconnected,              /* Buffer from driver.  */
                    sizeof(int),                /* Length of buffer in bytes. */
                    &dwReturned,                /* Bytes placed in DataBuffer. */
                    NULL                        /* NULL means wait till op. completes. */
                    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  
  if (fIoCtlSuccess && fIsDisconnected)
  {
    return BX_E_NOT_CONNECTED;
  }

  return BX_E_OK;
}

/* #endif */

/* --------------------------------------------------------------------------
 * 
 * -------------------------------------------------------------------------- */

bx_errtype EXPORT BestXCloseUSB(bx_portnumtype OsHandle)
{
  /* OS call to close device */
  if (!CloseHandle((HANDLE) OsHandle))
  {
    return BX_E_BAD_HANDLE;
  }

  return BX_E_OK;
}

/* --------------------------------------------------------------------------
 * Should only be called from BestXPortTimeoutSet()
 * -------------------------------------------------------------------------- */

bx_errtype BestXUSBPortTimeoutSet(bx_portnumtype OsHandle, 
                                BESTTIMEOUTS * pCallersTimeouts)
{
#if 0
  BOOL fIoCtlSuccess;
  ULONG ReturnedLength = 0;

  fIoCtlSuccess = DeviceIoControl(
    (HANDLE) OsHandle,          /* Handle to device  */
    (ULONG) IOCTL_USB_SET_TIMEOUT, /* IO Control code for setting timeouts */
    (PVOID)pCallersTimeouts,    /* Buffer to driver. */
    sizeof(BESTTIMEOUTS),       /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &ReturnedLength,            /* Bytes placed in DataBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  DBG_ApiFailureIsZero(fIoCtlSuccess);

  if (!fIoCtlSuccess) {
    return BX_E_API_ERROR;
  }
#endif  
  return BX_E_OK;
}

BOOL BestXDRR23(HANDLE OsHandle, unsigned long address, unsigned long size, unsigned long *val)
{
  VENDOR_REQUEST_IN  myRequest;
  BOOLEAN bresultOK = FALSE;
  unsigned long nBytes = 0;
  BYTE result=0;

  myRequest.bRequest  = ANCHOR_USB_DRR_0;
  myRequest.wValue    = ((WORD)(address>>16&0xffff)) | (WORD)(size==2?0:1<<15); 
  myRequest.wIndex    = (WORD)(address&0xffff);
  myRequest.wLength   = 1; /* dummy receive byte */
  myRequest.bData     = 0;     /* dummy */
  myRequest.direction = 0x01; /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
                                          IOCTL_Ezusb_VENDOR_REQUEST,
                                          &myRequest,
                                          sizeof(VENDOR_REQUEST_IN),
                                          &result,
                                          sizeof(BYTE),
                                          &nBytes,
                                          NULL);

  if (!bresultOK || nBytes!=1 || result != 0xa5)
  {
    return (BOOL) 0;
  }

  nBytes=0;
  result=0;
  myRequest.bRequest  = ANCHOR_USB_DRR_1;
  myRequest.wValue    = 0;      /* dummy */
  myRequest.wIndex    = 0;      /* dummy */
  myRequest.wLength   = sizeof(unsigned long);    
  myRequest.bData     = 0;      /* dummy */
  myRequest.direction = 0x01;   /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
                                          IOCTL_Ezusb_VENDOR_REQUEST,
                                          &myRequest,
                                          sizeof(VENDOR_REQUEST_IN),
                                          val,
                                          sizeof(unsigned long),
                                          &nBytes,
                                          NULL);

  if (!bresultOK || nBytes!=sizeof(unsigned long))
  {
    return (BOOL) 0;
  }

  return (BOOL)1;
}

BOOL BestXDRW23(HANDLE OsHandle, unsigned long address, unsigned long size, unsigned long val)
{
  VENDOR_REQUEST_IN  myRequest;
  BOOLEAN bresultOK = FALSE;
  unsigned long nBytes = 0;
  BYTE result=0;

  myRequest.bRequest  = ANCHOR_USB_DRW_0;
  myRequest.wValue    = ((WORD)(address>>16&0xffff)) | (WORD)(size==2?0:1<<15); 
  myRequest.wIndex    = (WORD)(address&0xffff);
  myRequest.wLength   = 1; /* dummy receive byte */    
  myRequest.bData     = 0;     /* dummy */
  myRequest.direction = 0x01;  /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
                                          IOCTL_Ezusb_VENDOR_REQUEST,
                                          &myRequest,
                                          sizeof(VENDOR_REQUEST_IN),
                                          &result,
                                          sizeof(BYTE),
                                          &nBytes,
                                          NULL);

  if (!bresultOK || nBytes!=1 || result!=0xa5)
  {
    return (BOOL)0;
  }

  nBytes=0;
  result=0;
  myRequest.bRequest  = ANCHOR_USB_DRW_1;
  myRequest.wValue    = (WORD)(val>>16 & 0xffff);
  myRequest.wIndex    = (WORD)(val & 0xffff);
  myRequest.wLength   = 1;    
  myRequest.bData     = 0;      /* dummy */
  myRequest.direction = 0x01;   /* IN */
          
  bresultOK = (BOOLEAN) DeviceIoControl (OsHandle,
               IOCTL_Ezusb_VENDOR_REQUEST,
               &myRequest,
               sizeof(VENDOR_REQUEST_IN),
               &result,
               1,
               &nBytes,
               NULL);

  if (!bresultOK||nBytes!=1||result!=0xa5)
  {
    return (BOOL)0;
  }

  return (BOOL)1;
}

